#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include "servdef.h"
#include "service.h"

static LONG ConvertErr(DWORD inerr)
{
		switch (inerr)
		{
		case (ERROR_ACCESS_DENIED):
			return SERVICE_ERROR_ACCESS_DENIED;
		case (ERROR_CIRCULAR_DEPENDENCY):
			return SERVICE_ERROR_CIRCULAR_DEPENDENCY;
		case (ERROR_DEPENDENT_SERVICES_RUNNING):
			return SERVICE_ERROR_DEPENDENT_SERVICES_RUNNING;
		case (ERROR_DUP_NAME):
			return SERVICE_ERROR_DUP_NAME;
		case (ERROR_FILE_NOT_FOUND):
			return SERVICE_ERROR_FILE_NOT_FOUND;
		case (ERROR_INSUFFICIENT_BUFFER ):
			return SERVICE_ERROR_INSUFFICIENT_BUFFER;
		case (ERROR_INVALID_HANDLE):
			return SERVICE_ERROR_INVALID_HANDLE;
		case (ERROR_INVALID_NAME):
			return SERVICE_ERROR_INVALID_NAME;
		case (ERROR_INVALID_PARAMETER):
			return SERVICE_ERROR_INVALID_PARAMETER;
		case (ERROR_INVALID_SERVICE_ACCOUNT):
			return SERVICE_ERROR_INVALID_SERVICE_ACCOUNT;
		case (ERROR_INVALID_SERVICE_CONTROL):
			return SERVICE_ERROR_INVALID_SERVICE_CONTROL;
		case (ERROR_INVALID_SERVICE_LOCK):
			return SERVICE_ERROR_INVALID_SERVICE_LOCK;
		case (ERROR_MORE_DATA):
			return SERVICE_ERROR_MORE_DATA;
		case (ERROR_PATH_NOT_FOUND):
			return SERVICE_ERROR_PATH_NOT_FOUND;
		case (ERROR_SERVICE_ALREADY_RUNNING):
			return SERVICE_ERROR_SERVICE_ALREADY_RUNNING;
		case (ERROR_SERVICE_CANNOT_ACCEPT_CTRL):
			return SERVICE_ERROR_SERVICE_CANNOT_ACCEPT_CTRL;
		case (ERROR_SERVICE_DATABASE_LOCKED):
			return SERVICE_ERROR_SERVICE_DATABASE_LOCKED;
		case (ERROR_SERVICE_DEPENDENCY_DELETED):
			return SERVICE_ERROR_SERVICE_DEPENDENCY_DELETED;
		case (ERROR_SERVICE_DEPENDENCY_FAIL):
			return SERVICE_ERROR_SERVICE_DEPENDENCY_FAIL;
		case (ERROR_SERVICE_DISABLED):
			return SERVICE_ERROR_SERVICE_DISABLED;
		case (ERROR_SERVICE_DOES_NOT_EXIST):
			return SERVICE_ERROR_SERVICE_DOES_NOT_EXIST;
		case (ERROR_SERVICE_EXISTS):
			return SERVICE_ERROR_SERVICE_EXISTS;
		case (ERROR_SERVICE_LOGON_FAILED):
			return SERVICE_ERROR_SERVICE_LOGON_FAILED;
		case (ERROR_SERVICE_MARKED_FOR_DELETE):
			return SERVICE_ERROR_SERVICE_MARKED_FOR_DELETE;
		case (ERROR_SERVICE_NOT_ACTIVE):
			return SERVICE_ERROR_SERVICE_NOT_ACTIVE;
		case (ERROR_SERVICE_NO_THREAD):
			return SERVICE_ERROR_SERVICE_NO_THREAD;
		case (ERROR_SERVICE_REQUEST_TIMEOUT):
			return SERVICE_ERROR_SERVICE_REQUEST_TIMEOUT;

		default:
			return SERVICE_ERROR_UNKNOWN;
		}

}


// InstallChangeServiceConfig
//
// This function changes a service configuration as per
// ChangeServiceConfig and returns SERVICE_SUCCESS if successful 
// otherwise this function returns the appropriate
// error code from one of the following:
//
//			SERVICE_ERROR_ACCESS_DENIED;
//			SERVICE_ERROR_CIRCULAR_DEPENDENCY;
//			SERVICE_ERROR_DUP_NAME;
//			SERVICE_ERROR_INVALID_HANDLE;
//			SERVICE_ERROR_INVALID_PARAMETER;
//			SERVICE_ERROR_INVALID_SERVICE_ACCOUNT;
//			SERVICE_ERROR_SERVICE_MARKED_FOR_DELETE;

_declspec(dllexport) LONG InstallChangeServiceConfig(SC_HANDLE hService,
													 DWORD dwServiceType,
													 DWORD dwStartType,
													 DWORD dwErrorControl, 
													 LPCTSTR lpBinaryPathName, 
													 LPCTSTR lpLoadOrderGroup, 
													 LPDWORD lpdwTagId, 
													 LPCTSTR lpDependencies, 
													 LPCTSTR lpServiceStartName, 
													 LPCTSTR lpPassword, 
													 LPCTSTR lpDisplayName)
{
	if (!ChangeServiceConfig(hService,
		 dwServiceType,
		 dwStartType,
		 dwErrorControl, 
		 lpBinaryPathName, 
		 lpLoadOrderGroup, 
		 lpdwTagId, 
		 lpDependencies, 
		 lpServiceStartName, 
		 lpPassword, 
		 lpDisplayName))
	{
		return(ConvertErr(GetLastError()));
	}

	return SERVICE_SUCCESS;
}


// InstallCloseServiceHandle
//
// This function closes a service as per CloseService
// and returns SERVICE_SUCCESS if successful other wise
// this function returns the appropriate
// error code from one of the following:
//
//			SERVICE_ERROR_ACCESS_DENIED;
//			SERVICE_ERROR_INVALID_HANDLE;
//			SERVICE_ERROR_INVALID_NAME;
//			SERVICE_ERROR_SERVICE_EXISTS;

_declspec(dllexport) LONG InstallCloseServiceHandle(SC_HANDLE hService)
{
	if (!CloseServiceHandle(hService))
	{
		return(ConvertErr(GetLastError()));
	}

	return SERVICE_SUCCESS;
}


// InstallControlService
//
// This function starts a Service as per StartService
// and returns SERVICE_SUCCESS if successful otherwise it returns 
// the appropriate error code from one of the following:
//
//			SERVICE_ERROR_ACCESS_DENIED;
//			SERVICE_ERROR_SERVICE_DATABASE_LOCKED;
//			SERVICE_ERROR_DEPENDENT_SERVICES_RUNNING;
//			SERVICE_ERROR_INVALID_SERVICE_CONTROL;
//			SERVICE_ERROR_SERVICE_CANNOT_ACCEPT_CTRL;
//			SERVICE_ERROR_SERVICE_NOT_ACTIVE;
//			SERVICE_ERROR_SERVICE_REQUEST_TIMEOUT;

_declspec(dllexport) LONG InstallControlService(SC_HANDLE hService,
											   DWORD dwControl, 
											   LPSERVICE_STATUS lpStatus)
{
	if (!ControlService(hService, dwControl, lpStatus))
	{
		return(ConvertErr(GetLastError()));
	}
	
	return SERVICE_SUCCESS;
}


// InstallCreateService
//
// This function creates a Service as per CreateService
// and returns SERVICE_SUCCESS if successful otherwise it returns
// the appropriate error code of the following:
//
//			SERVICE_ERROR_ACCESS_DENIED;
//			SERVICE_ERROR_CIRCULAR_DEPENDENCY;
//			SERVICE_ERROR_DUP_NAME;
//			SERVICE_ERROR_INVALID_HANDLE;
//			SERVICE_ERROR_INVALID_NAME;
//			SERVICE_ERROR_INVALID_PARAMETER;
//			SERVICE_ERROR_INVALID_SERVICE_ACCOUNT;
//			SERVICE_ERROR_SERVICE_EXISTS;

_declspec(dllexport) LONG InstallCreateService(SC_HANDLE hSCManager,
										 	   LPCTSTR lpServiceName, 
											   LPCTSTR lpDisplayName,
											   DWORD dwDesiredAccess,
											   DWORD dwServiceType,
											   DWORD dwStartType,
											   DWORD dwErrorControl,
											   LPCTSTR lpBinaryPathName,
											   LPCTSTR lpLoadOrderGroup,
											   LPDWORD lpdwTagId,
											   LPCTSTR lpDependencies,
											   LPCTSTR lpServiceStartName,
											   LPCTSTR lpPassword,
											   LPSC_HANDLE lphService)
{
	char	*multizspace;

	if (lpDependencies)
		multizspace = malloc(strlen(lpDependencies)+1);
	else
		multizspace = NULL;

	if ( (multizspace) || (!lpDependencies) )
	{
		unsigned int	l, x;

		if (lpDependencies)
		{
			strcpy(multizspace, lpDependencies);
			l	=	strlen(multizspace);
			for (x=0; x < l; x++)
				if (multizspace[x] == ',')
					multizspace[x] = 0;
		}

		*lphService = CreateService (hSCManager,
									 lpServiceName, 
									 lpDisplayName,
									 dwDesiredAccess,
									 dwServiceType,
									 dwStartType,
									 dwErrorControl,
									 lpBinaryPathName,
									 lpLoadOrderGroup,
									 lpdwTagId,
									 multizspace, 
									 lpServiceStartName, 
									 lpPassword
									 );
		if (multizspace)
			free(multizspace);

		if (*lphService == NULL)
		{
			return(ConvertErr(GetLastError()));
		}

		return SERVICE_SUCCESS;
	}
	else
		return(SERVICE_ERROR_UNKNOWN);
}


// InstallDeleteService
//
// This function deletes a Service as per DeleteService
// and returns SERVICE_SUCCESS if successful otherwise it returns
// the appropriate error code from one of the following:
//
//			SERVICE_ERROR_ACCESS_DENIED;
//			SERVICE_ERROR_INVALID_HANDLE;
//			SERVICE_ERROR_SERVICE_MARKED_FOR_DELETE;

_declspec(dllexport) LONG InstallDeleteService(SC_HANDLE hService)
{
	
	// delete service	
	if (!DeleteService(hService))
	{
		return(ConvertErr(GetLastError()));
	}

    return SERVICE_SUCCESS;
}


// InstallEnumDependentServices
//
// This function enumerates the dependent services of a service as per
// EnumDependantServices and returns SERVICE_SUCCESS if successful 
// otherwise this function returns the appropriate
// error code from one of the following:
//
//			SERVICE_ERROR_ACCESS_DENIED;
//			SERVICE_ERROR_INVALID_HANDLE;
//			SERVICE_ERROR_INVALID_PARAMETER;
//			SERVICE_ERROR_MORE_DATA;

_declspec(dllexport) LONG InstallEnumDependentServices(SC_HANDLE hService, 
													   DWORD dwServiceState, 
													   LPENUM_SERVICE_STATUS lpServices, 
													   DWORD cbBufSize, 
													   LPDWORD pcbBytesNeeded, 
													   LPDWORD lpServicesReturned)
{
	if (!EnumDependentServices(hService, 
	   dwServiceState, 
	   lpServices, 
	   cbBufSize, 
	   pcbBytesNeeded, 
	   lpServicesReturned))
	{
		return(ConvertErr(GetLastError()));
	}
	
	return SERVICE_SUCCESS;
}


// InstallEnumServicesStatus
//
// This function enumerates status's of all services as per
// EnumServicesStatus and returns SERVICE_SUCCESS if successful 
// otherwise this function returns the appropriate
// error code from one of the following:
//
//			SERVICE_ERROR_ACCESS_DENIED;
//			SERVICE_ERROR_INVALID_HANDLE;
//			SERVICE_ERROR_INVALID_PARAMETER;
//			SERVICE_ERROR_MORE_DATA;

_declspec(dllexport) LONG InstallEnumServicesStatus(SC_HANDLE hSCManager, 
													DWORD dwServiceType, 
													DWORD dwServiceState, 
													LPENUM_SERVICE_STATUS lpServices, 
													DWORD cbBufSize, 
													LPDWORD pcbBytesNeeded, 
													LPDWORD lpServicesReturned, 
													LPDWORD lpResumeHandle)
{
	if (!EnumServicesStatus(hSCManager, 
		dwServiceType, 
		dwServiceState, 
		lpServices, 
		cbBufSize, 
		pcbBytesNeeded, 
		lpServicesReturned, 
		lpResumeHandle))
	{
		return(ConvertErr(GetLastError()));
	}
	
	return SERVICE_SUCCESS;
}


// InstallGetServiceDisplayName
//
// This function gets the display name of a service as per
// GetServiceDisplayName and returns SERVICE_SUCCESS if successful 
// otherwise this function returns the appropriate
// error code from one of the following:
//
//			SERVICE_ERROR_ACCESS_DENIED;
//			SERVICE_ERROR_INVALID_HANDLE;
//			SERVICE_ERROR_INVALID_NAME;

_declspec(dllexport) LONG InstallGetServiceDisplayName(SC_HANDLE hSCManager, 
													   LPCTSTR lpServiceName, 
													   LPTSTR lpDisplayName, 
													   LPDWORD lpcchBuffer)
{
	if (!GetServiceDisplayName(hSCManager, 
	   lpServiceName, 
	   lpDisplayName, 
	   lpcchBuffer))
	{
		return(ConvertErr(GetLastError()));
	}

	return SERVICE_SUCCESS;
}


// InstallGetServiceKeyName
//
// This function gets the key name of a service as per
// GetServiceKeyName and returns SERVICE_SUCCESS if successful 
// otherwise this function returns the appropriate
// error code from one of the following:
//
//			SERVICE_ERROR_ACCESS_DENIED;
//			SERVICE_ERROR_INVALID_HANDLE;
//			SERVICE_ERROR_INVALID_NAME;

_declspec(dllexport) LONG InstallGetServiceKeyName(SC_HANDLE hSCManager,
												   LPCTSTR lpDisplayName, 
												   LPTSTR lpServiceName, 
												   LPDWORD lpcchBuffer)
{
	if (!GetServiceKeyName(hSCManager,
	   lpDisplayName, 
	   lpServiceName, 
	   lpcchBuffer))
	{
		return(ConvertErr(GetLastError()));
	}

	return SERVICE_SUCCESS;
}


// InstallLockServiceDatabase
//
// This function locks the service database as per
// LockServiceDatabase and returns SERVICE_SUCCESS if successful 
// otherwise this function returns the appropriate
// error code from one of the following:
//
//			SERVICE_ERROR_ACCESS_DENIED;
//			SERVICE_ERROR_INVALID_HANDLE;
//			SERVICE_ERROR_SERVICE_DATABASE_LOCKED

_declspec(dllexport) LONG InstallLockServiceDatabase(SC_HANDLE hSCManager, SC_LOCK* lphLock){
	*lphLock = LockServiceDatabase(hSCManager);
	
	if (*lphLock == NULL)
	{
		return(ConvertErr(GetLastError()));
	}

	return SERVICE_SUCCESS;
}



// InstallOpenSCManager
//
// This function opens the local active service manager as per OpenSCManager
// and returns SERVICE_SUCCESS if successful otherwise it returns the appropriate
// error code from one of the following:
//
//			SERVICE_ERROR_ACCESS_DENIED;
//			SERVICE_ERROR_DATABASE_DOES_NOT_EXIST
//			SERVICE_ERROR_INVALID_PARAMETER
//
// NOTE: only the local active service manager is supported because microsoft
// is too STUPID to provide any way of querying the manager once it has been 
// opened thereby making it impossible to provide the reverse action of a close

_declspec(dllexport) LONG InstallOpenSCManager(LPCTSTR lpMachineName, 
											   LPCTSTR lpDatabaseName,
											   DWORD dwDesiredAccess,
											   LPSC_HANDLE lphSCManager)
{
	
	*lphSCManager = OpenSCManager(	lpMachineName,
									lpDatabaseName, 
									dwDesiredAccess);

	if (*lphSCManager == NULL)
	{
		return(ConvertErr(GetLastError()));
	}
	
	return SERVICE_SUCCESS;
}

// InstallOpenService
//
// This function opens a service as per OpenService
// and returns SERVICE_SUCCESS if successful otherwise it returns
// the appropriate error code of the following:
//
//			SERVICE_ERROR_ACCESS_DENIED;
//			SERVICE_ERROR_INVALID_HANDLE;
//			SERVICE_ERROR_INVALID_NAME;
//			SERVICE_ERROR_SERVICE_EXISTS;

_declspec(dllexport) LONG InstallOpenService(SC_HANDLE hSCManager,
											 LPCTSTR lpServiceName,
											 DWORD dwDesiredAccess,
											 LPSC_HANDLE lphService)
{
	*lphService = OpenService(	hSCManager,
								lpServiceName, 
								dwDesiredAccess);

	if (*lphService == NULL)
	{
		return(ConvertErr(GetLastError()));
	}
	
	return SERVICE_SUCCESS;
}


// InstallQueryServiceConfig
//
// This function queries a service as per QueryServiceConfig
// and returns SERVICE_SUCCESS if successful otherwise
// this function returns the appropriate
// error code from one of the following:
//
//			SERVICE_ERROR_ACCESS_DENIED;
//			SERVICE_ERROR_INSUFFICIENT_BUFFER;
//			SERVICE_ERROR_INVALID_HANDLE;

_declspec(dllexport) LONG InstallQueryServiceConfig(SC_HANDLE hService, 
													LPQUERY_SERVICE_CONFIG lpServiceConfig, 
													DWORD cbBufSize, 
													LPDWORD pcbBytesNeeded)
{
	if (!QueryServiceConfig(hService, lpServiceConfig, cbBufSize, pcbBytesNeeded))
	{
		return(ConvertErr(GetLastError()));
	}

	return SERVICE_SUCCESS;
}


// InstallQueryServiceLockStatus
//
// This function queries the lock status of the service database as per
// QueryServiceLockStatus and returns SERVICE_SUCCESS if successful 
// otherwise this function returns the appropriate
// error code from one of the following:
//
//			SERVICE_ERROR_ACCESS_DENIED;
//			SERVICE_ERROR_INSUFFICIENT_BUFFER;
//			SERVICE_ERROR_INVALID_HANDLE;

_declspec(dllexport) LONG InstallQueryServiceLockStatus(SC_HANDLE hSCManager, 
														LPQUERY_SERVICE_LOCK_STATUS lpLockStatus, 
														DWORD cbBufSize, 
														LPDWORD pcbBytesNeeded)
{
	if (!QueryServiceLockStatus(hSCManager, 
		lpLockStatus, 
		cbBufSize, 
		pcbBytesNeeded))
	{
		return(ConvertErr(GetLastError()));
	}

	return SERVICE_SUCCESS;
}


// InstallQueryServiceObjectSecurity
//
// This function queries a service's security as per
// QueryServiceObjectSecurity and returns SERVICE_SUCCESS if successful 
// otherwise this function returns the appropriate
// error code from one of the following:
//
//			SERVICE_ERROR_ACCESS_DENIED;
//			SERVICE_ERROR_INSUFFICIENT_BUFFER;
//			SERVICE_ERROR_INVALID_HANDLE;
//			SERVICE_ERROR_INVALID_PARAMETER;

_declspec(dllexport) LONG InstallQueryServiceObjectSecurity(SC_HANDLE hService, 
															SECURITY_INFORMATION dwSecurityInformation, 
															PSECURITY_DESCRIPTOR lpSecurityDescriptor, 
															DWORD cbBufSize, 
															LPDWORD pcbBytesNeeded)
{
	if (!QueryServiceObjectSecurity(hService, 
		dwSecurityInformation, 
		lpSecurityDescriptor, 
		cbBufSize, 
		pcbBytesNeeded))
	{
		return(ConvertErr(GetLastError()));
	}

	return SERVICE_SUCCESS;
}


// InstallQueryServiceStatus
//
// This function queries a service's status as per
// QueryServiceStatus and returns SERVICE_SUCCESS if successful 
// otherwise this function returns the appropriate
// error code from one of the following:
//
//			SERVICE_ERROR_ACCESS_DENIED;
//			SERVICE_ERROR_INVALID_HANDLE;

_declspec(dllexport) LONG InstallQueryServiceStatus(SC_HANDLE hService, 
													LPSERVICE_STATUS lpServiceStatus)
{
	if (!QueryServiceStatus(hService, 
		lpServiceStatus))
	{
		return(ConvertErr(GetLastError()));
	}

	return SERVICE_SUCCESS;
}


// InstallSetServiceObjectSecurity
//
// This function sets a service's security as per
// SetServiceObjectSecurity and returns SERVICE_SUCCESS if successful 
// otherwise this function returns the appropriate
// error code from one of the following:
//
//			SERVICE_ERROR_ACCESS_DENIED;
//			SERVICE_ERROR_INVALID_HANDLE;
//			SERVICE_ERROR_INVALID_PARAMETER;
//			SERVICE_ERROR_SERVICE_MARKED_FOR_DELETE;

_declspec(dllexport) LONG InstallSetServiceObjectSecurity(SC_HANDLE hService, 
														  SECURITY_INFORMATION dwSecurityInformation, 
														  PSECURITY_DESCRIPTOR lpSecurityDescriptor )
{
	if (!SetServiceObjectSecurity(hService, 
	  dwSecurityInformation, 
	  lpSecurityDescriptor))
	{
		return(ConvertErr(GetLastError()));
	}

	return SERVICE_SUCCESS;
}


// InstallStartService
//
// This function starts a Service as per StartService
// and returns SERVICE_SUCCESS if successful otherwise it returns
// the appropriate error code from one of the following:
//
//			SERVICE_ERROR_ACCESS_DENIED;
//			SERVICE_ERROR_INVALID_HANDLE;
//			SERVICE_ERROR_PATH_NOT_FOUND;
//			SERVICE_ERROR_SERVICE_ALREADY_RUNNING;
//			SERVICE_ERROR_SERVICE_DATABASE_LOCKED;
//			SERVICE_ERROR_SERVICE_DEPENDENCY_DELETED;
//			SERVICE_ERROR_SERVICE_DEPENDENCY_FAIL;
//			SERVICE_ERROR_SERVICE_DISABLED;
//			SERVICE_ERROR_SERVICE_LOGON_FAILED;
//			SERVICE_ERROR_SERVICE_MARKED_FOR_DELETE;
//			SERVICE_ERROR_SERVICE_NO_THREAD;
//			SERVICE_ERROR_SERVICE_REQUEST_TIMEOUT;

_declspec(dllexport) LONG InstallStartService( SC_HANDLE hService,
											   DWORD dwNumServiceArgs,
											   LPCTSTR *lpServiceArgVectors
											   )
{
	if (!StartService(hService, dwNumServiceArgs, lpServiceArgVectors))
	{
		return(ConvertErr(GetLastError()));
	}

	return SERVICE_SUCCESS;
}


// InstallUnlockServiceDatabase
//
// This function unlocks the service database as per
// UnlockServiceDatabase and returns SERVICE_SUCCESS if successful 
// otherwise this function returns the appropriate
// error code from one of the following:
//

_declspec(dllexport) LONG InstallUnlockServiceDatabase(SC_LOCK ScLock)
{
	if (!UnlockServiceDatabase(ScLock))
	{
		return(ConvertErr(GetLastError()));
	}

	return SERVICE_SUCCESS;
}
