// InstallSvc.c
//
// Functions to install and uninstall the system service.

#include <windows.h>
#include <tchar.h>

#define SERVICE_NAME  _T("MYSERVICE")
#define DISPLAY_NAME  _T("MyService")

///////////////////////////////////////////////////////////////////////////////

_declspec(dllexport) DWORD InstallService(LPCTSTR szExeName, LPCTSTR szAcctName,
										  LPCTSTR szPassword, BOOL bAutoStart)
{
	_TCHAR    szImagePath[MAX_PATH];
	_TCHAR    szLongAcctName[32];
	SC_HANDLE hSCManager = NULL;
	SC_HANDLE hService = NULL;
	DWORD     dwStartType = bAutoStart ? SERVICE_AUTO_START : SERVICE_DEMAND_START;
	DWORD     rc;

	
	// Validate parameters
	if (!szExeName || !*szExeName)
	{
		return ERROR_INVALID_PARAMETER;
	}

	if (szAcctName && !*szAcctName)
	{
		szAcctName = NULL;	// Treat "" account same as NULL.
		szPassword = NULL;
	}

	// Generate name of service executable
	if (!SearchPath(NULL, szExeName, _T(".exe"), MAX_PATH, szImagePath, NULL))
	{
		return GetLastError();
	}

	// Add local domain name to account if no domain supplied
	if (szAcctName && !_tcschr(szAcctName, _T('\\')))
	{
		_tcscpy(szLongAcctName, _T(".\\"));
		_tcscat(szLongAcctName, szAcctName);
		szAcctName = szLongAcctName;
	}

	if ((hSCManager = OpenSCManager(NULL, NULL, SC_MANAGER_CREATE_SERVICE)) == NULL)
	{
		return GetLastError();
	}

	// Either update an existing service, or create a new one.
	if ((hService = OpenService(hSCManager, SERVICE_NAME, SERVICE_CHANGE_CONFIG)) == NULL)
	{
		if ((rc = GetLastError()) != ERROR_SERVICE_DOES_NOT_EXIST)
		{
			CloseServiceHandle(hSCManager);
			return rc;
		}

		// Create new service
		if ((hService = CreateService(hSCManager, SERVICE_NAME, DISPLAY_NAME,
									  SERVICE_ALL_ACCESS, SERVICE_WIN32_OWN_PROCESS,
									  dwStartType, SERVICE_ERROR_IGNORE,
									  szImagePath, NULL, NULL, NULL, szAcctName, szPassword)) == NULL)
		{
			rc = GetLastError();
			CloseServiceHandle(hSCManager);
			return rc;
		}
		else
		{
			SECURITY_DESCRIPTOR SecurityDescriptor;

			// Set service's security descriptor to allow all access
			InitializeSecurityDescriptor(&SecurityDescriptor, SECURITY_DESCRIPTOR_REVISION);
			SetSecurityDescriptorDacl(&SecurityDescriptor, TRUE, NULL, FALSE);

			if (!SetServiceObjectSecurity(hService, DACL_SECURITY_INFORMATION,
										  &SecurityDescriptor))
			{
				rc = GetLastError();
				CloseServiceHandle(hService);
				CloseServiceHandle(hSCManager);
				return rc;
			}
		}
	}
	else
	{
		// Update configuration of existing service

		// If szAcctName is NULL, this means that we want to change the service
		// account to the LocalSystem account.
		if (!szAcctName)
		{
			szAcctName = ".\\LocalSystem";
			szPassword = "";		// Need this or won't work
		}

		if (!ChangeServiceConfig(hService, SERVICE_WIN32_OWN_PROCESS, dwStartType,
								 SERVICE_NO_CHANGE, szImagePath, NULL, NULL, NULL,
								 szAcctName, szPassword, DISPLAY_NAME))
		{
			rc = GetLastError();
			CloseServiceHandle(hService);
			CloseServiceHandle(hSCManager);
			return rc;
		}
	}

	CloseServiceHandle(hService);
	CloseServiceHandle(hSCManager);

	return 0;
}


_declspec(dllexport) DWORD UninstallService(void)
{
	SC_HANDLE hSCManager = NULL;
	SC_HANDLE hService = NULL;
	DWORD     rc = 0;

	if ((hSCManager = OpenSCManager(NULL, NULL, SC_MANAGER_CREATE_SERVICE)) == NULL)
	{
		rc = GetLastError();
	}
	else if ((hService = OpenService(hSCManager, SERVICE_NAME, DELETE)) == NULL)
	{
		rc = GetLastError();
	}
	else if (!DeleteService(hService))
	{
		rc = GetLastError();
	}

	if (hService)
		CloseServiceHandle(hService);
	if (hSCManager)
		CloseServiceHandle(hSCManager);

	return rc;
}

// UninstInitialize, UninstUnInitialize
//
// These functions are called by the InstallShield uninstaller.  UninstUnInitialize
// is called at the end of the uninstallation and removes the AAI service entry from
// the Service Manager database.

_declspec(dllexport) long UninstInitialize(HWND hWnd, HINSTANCE hInst, long Reserved)
{
	// Continue with uninstall
	return 0;
}

_declspec(dllexport) long UninstUnInitialize(HWND hWnd, HINSTANCE hInst, long Reserved)
{
	// Delete AAI service
	UninstallService();

	return 0;
}
